package ga.view.streaming.nodes;

import ga.core.individual.ICostInfo;
import ga.core.individual.IDebugInfo;
import ga.core.individual.IIndividual;
import ga.view.processor.OffscreenProcessor;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;

import java.util.logging.Logger;

import com.jme3.asset.AssetManager;
import com.jme3.bullet.collision.PhysicsCollisionObject;
import com.jme3.bullet.collision.shapes.BoxCollisionShape;
import com.jme3.bullet.control.RigidBodyControl;
import com.jme3.material.Material;
import com.jme3.material.RenderState.BlendMode;
import com.jme3.math.ColorRGBA;
import com.jme3.math.FastMath;
import com.jme3.math.Quaternion;
import com.jme3.math.Vector3f;
import com.jme3.renderer.Camera;
import com.jme3.renderer.queue.RenderQueue.Bucket;
import com.jme3.renderer.queue.RenderQueue.ShadowMode;
import com.jme3.scene.Geometry;
import com.jme3.scene.Node;
import com.jme3.scene.shape.Box;
import com.jme3.system.AppSettings;

/**
 * This is a node that displays a show room as texture.
 * 
 * @param <T>
 *          The generic type of the individuals.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class PanelNode<T extends IIndividual<T>> extends Node {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(PanelNode.class
      .getName());

  private static final Vector3f INSPECT_LOCATION = new Vector3f(0f, 7.3f, 0f);

  /**
   * Type of node info to display.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public enum InfoStringType {
    COSTS, ID, DEBUG, GENOTYPE, NONE
  }

  private AnchorNode<T> anchor;
  private final float x = 1.5f;
  private final float y = 0f;
  private final float z;
  private final Material mat;

  private float materialAlpha = -1.0f;
  private float fadeToAlpha = 1.0f;

  private final OffscreenProcessor processor;
  private final ShowRoom showRoom;

  private float fadeSpeed = 1.0f;

  private PanelNodeListener<T> listener;

  private boolean isInitFade;

  private Vector3f targetLocation;
  private Vector3f lastLocation;

  private Mode mode = Mode.NORMAL;

  // this is the evaluation stuff
  private final T individual;
  private final RigidBodyControl control;

  private InfoStringType infoStringType = InfoStringType.NONE;

  /**
   * The state of the panel node.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public enum Mode {
    NORMAL, FADE, MOVE_TO_INSPECT, INSPECT, INSPECT_DONE
  }

  /**
   * Instantiates a new panel node.
   * 
   * @param assetManager
   *          the asset manager
   * @param settings
   *          the settings
   * @param showRoom
   *          the show room
   * @param camSettings
   *          the cam settings
   * @param individual
   *          the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public PanelNode(final AssetManager assetManager, final AppSettings settings,
      final ShowRoom showRoom, final CameraSettings camSettings,
      final T individual) {
    super("Panel Node");
    this.showRoom = showRoom;
    final float aspectRatio = settings.getWidth()
        / (float) settings.getHeight();
    z = x / aspectRatio;
    this.individual = individual;
    final Material offscreenMat = new Material(assetManager,
        "Common/MatDefs/Light/Lighting.j3md");
    offscreenMat.setFloat("Shininess", 0.0f);
    offscreenMat.setBoolean("UseMaterialColors", true);
    offscreenMat.setBoolean("UseAlpha", true);
    offscreenMat.setColor("Ambient", ColorRGBA.Black);
    offscreenMat.setColor("Diffuse", ColorRGBA.White);
    offscreenMat.setColor("Specular", ColorRGBA.Black);
    offscreenMat.setReceivesShadows(true);

    offscreenMat.getAdditionalRenderState().setBlendMode(BlendMode.Alpha);

    processor = new OffscreenProcessor(assetManager, settings,
        showRoom.clone(true), offscreenMat, "DiffuseMap");

    processor.setTextureSize(1024, 1024);

    updateDebugText();

    // TODO find a better way to do this
    final Camera cam = processor.getCamera();

    camSettings.configureCamera(cam);

    this.mat = processor.getMaterial();

    // geometry
    final Box box = new Box(z, y, x);
    final Geometry boxGeo = new Geometry("panel", box);
    boxGeo.setMaterial(mat);
    boxGeo.rotate(0f, FastMath.DEG_TO_RAD * -90f, 0f);
    attachChild(boxGeo);

    // collision
    control = new RigidBodyControl(new BoxCollisionShape(
        new Vector3f(x, .1f, z)));

    addControl(control);

    setShadowMode(ShadowMode.Off);
    control.setCollisionGroup(PhysicsCollisionObject.COLLISION_GROUP_01);
    control.setMass(50f);
    control.setFriction(.01f);
    control.setLinearDamping(.2f);

    // damp angular impulses
    control.setAngularDamping(1f);

    // never sleep
    control.setLinearSleepingThreshold(0f);

    setQueueBucket(Bucket.Transparent);
  }

  /**
   * Sets the info string type.
   * 
   * @param infoStringType
   *          the new info string type
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setInfoStringType(final InfoStringType infoStringType) {
    this.infoStringType = infoStringType;

    updateDebugText();
  }

  /**
   * Updates debug text.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void updateDebugText() {
    String debugText = String.valueOf(individual.getId());

    switch (infoStringType) {
    case COSTS:
      if (individual instanceof ICostInfo) {
        debugText = ((ICostInfo) individual).getCostString();
      }
      break;
    case DEBUG:
      if (individual instanceof IDebugInfo) {
        debugText = ((IDebugInfo) individual).getDebugString();
      }
      break;
    case ID:
      if (individual instanceof IDebugInfo) {
        debugText = ((IDebugInfo) individual).getIdString();
      }
      break;
    case GENOTYPE:
      if (individual instanceof IDebugInfo) {
        debugText = ((IDebugInfo) individual).getGenotypeString();
      }
      break;
    case NONE:
    default:
      debugText = "";
      break;
    }

    processor.setText(debugText);
  }

  /**
   * Sets the camera settings.
   * 
   * @param camSettings
   *          the new camera settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setCameraSettings(final CameraSettings camSettings) {
    final Camera cam = processor.getCamera();

    camSettings.configureCamera(cam);
  }

  /**
   * Gets the control.
   * 
   * @return the control
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public RigidBodyControl getControl() {
    return control;
  }

  /**
   * Gets the show room.
   * 
   * @return the show room
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ShowRoom getShowRoom() {
    return showRoom;
  }

  @Override
  public void updateLogicalState(final float tpf) {
    super.updateLogicalState(tpf);

    if (mode == Mode.MOVE_TO_INSPECT || mode == Mode.INSPECT_DONE) {
      boolean inPosition = true;

      final Vector3f loc = getLocalTranslation();

      final float distance = targetLocation.distance(loc);

      if (distance > .05f) {
        inPosition = false;

        // the move step should be limited by the distance to the target
        final float maxMoveStep = Math.min(distance, tpf * 4f);

        // calc the delta translation by subtracting the direction vector
        // multiplied with the step size from the target location
        final Vector3f dir = targetLocation.subtract(loc).normalize()
            .mult(maxMoveStep);
        setLocalTranslation(loc.add(dir));
      }

      final Quaternion rot = getLocalRotation();
      float d = 0f;
      final float diff = 0.002f;
      final float factor = 0.001f;

      float dirX = rot.getX();
      if (FastMath.abs(dirX - d) > diff) {
        inPosition = false;
        dirX += d - (dirX + dirX * factor);
        LOGGER.info("X: " + rot.getX());
      }

      d = 0;
      float dirY = rot.getY();
      if (FastMath.abs(dirY - d) > diff) {
        inPosition = false;
        dirY += d - (dirY + dirY * factor);
        LOGGER.info("Y: " + rot.getY());
      }

      d = 0;
      float dirZ = rot.getZ();
      if (FastMath.abs(dirZ - d) > diff) {
        inPosition = false;
        dirZ += d - (dirZ + dirZ * factor);
        LOGGER.info("Z: " + rot.getZ());
      }

      d = 1;
      float dirW = rot.getW();
      if (FastMath.abs(dirW - d) > diff) {
        inPosition = false;
        dirW += d - (dirW + dirW * factor);
        LOGGER.info("W: " + rot.getW());
      }

      rot.set(dirX, dirY, dirZ, dirW);

      if (inPosition) {
        if (mode == Mode.MOVE_TO_INSPECT) {
          mode = Mode.INSPECT;

          if (listener != null) {
            listener.panelInInspectPosition(this);
          }
        } else if (mode == Mode.INSPECT_DONE) {
          mode = Mode.NORMAL;
          if (control != null) {
            control.setEnabled(true);
            control.setLinearDamping(.2f);
          }
          if (getAnchor() != null) {
            getAnchor().attachJoint(this);
          }

        }
      }
    } else if (mode == Mode.INSPECT) {
      targetLocation = lastLocation;
    }

    if (Math.abs(materialAlpha - fadeToAlpha) > .000001f) {
      // bigger
      if (materialAlpha > fadeToAlpha) {
        materialAlpha -= tpf * fadeSpeed;

        if (materialAlpha < fadeToAlpha) {
          materialAlpha = fadeToAlpha;
        }

        // less
      } else {
        materialAlpha += tpf * fadeSpeed;
        if (materialAlpha > fadeToAlpha) {
          materialAlpha = fadeToAlpha;
          isInitFade = false;
        }
      }

      mat.setColor("Diffuse", new ColorRGBA(1f, 1f, 1f, materialAlpha));
    }

    if (!isInitFade && Math.abs(materialAlpha) < .000001f && listener != null) {
      listener.panelReadyToDestroy(this);
      listener = null;
    }
  }

  /**
   * Triggers an inspect event. The node will move to the camera.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void inspect() {
    mode = Mode.MOVE_TO_INSPECT;
    if (getAnchor() != null) {
      getAnchor().detachJoint();
    }
    if (control != null) {
      control.setEnabled(false);
      control.setGravity(Vector3f.ZERO);
      control.setLinearDamping(2f);
    }
    targetLocation = INSPECT_LOCATION;
    lastLocation = getLocalTranslation();
  }

  /**
   * Triggers an inspect done event. The node will move back to the old
   * position.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void inspectDone() {
    this.mode = Mode.INSPECT_DONE;
  }

  @Override
  protected void setParent(final Node parent) {
    super.setParent(parent);
    control.setPhysicsLocation(this.getLocalTranslation());
  }

  /**
   * Gets the individual.
   * 
   * @return the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public T getIndividual() {
    return individual;
  }

  /**
   * Sets the panel node listener.
   * 
   * @param listener
   *          the new panel node listener
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setPanelNodeListener(final PanelNodeListener<T> listener) {
    this.listener = listener;
  }

  /**
   * Sets the anchor.
   * 
   * @param anchor
   *          the new anchor
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setAnchor(final AnchorNode<T> anchor) {
    this.anchor = anchor;
  }

  /**
   * Gets the anchor.
   * 
   * @return the anchor
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public AnchorNode<T> getAnchor() {
    return anchor;
  }

  /**
   * Gets the offscreen processor.
   * 
   * @return the processor
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public OffscreenProcessor getProcessor() {
    return processor;
  }

  /**
   * Triggers fading to a given alpha value.
   * 
   * @param alpha
   *          the new alpha
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFadeToAlpha(final float alpha) {
    this.fadeToAlpha = alpha;
  }

  /**
   * Sets the fade speed.
   * 
   * @param fadeSpeed
   *          the new fade speed
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFadeSpeed(final float fadeSpeed) {
    this.fadeSpeed = fadeSpeed;
  }
}
